

################################################################################



#' Box-Cox function
#'
#' Create Box-Cox transformation function(s) for value(s) of \sQuote{y}.
#'
#' @param y Object convertible to a numeric vector.
#' @param simplify Logical. Return function if resulting list contained only
#'   a single value, \code{NULL} if it contains no values?
#' @inheritParams box_cox
#' @family transformation-functions
#' @keywords manip
#' @export
#' @return List of functions, or single function, or \code{NULL}.
#' @references \url{http://en.wikipedia.org/wiki/Box-Cox_transformation}
#' @references Legendre, P., Legendre, L. 1998 \emph{Numerical Ecology.}
#'   Amsterdam: Elsevier, 853 p.
#' @examples
#' (x <- box_cox_fun(0:2))
#' stopifnot(is.list(x), length(x) == 3, sapply(x, class) == "function")
#'
box_cox_fun <- function(y, ...) UseMethod("box_cox_fun")

#' @rdname box_cox_fun
#' @method box_cox_fun default
#' @export
#'
box_cox_fun.default <- function(y, simplify = TRUE, ...) {
  box_cox_fun(as.numeric(y), simplify = simplify, ...)
}

#' @rdname box_cox_fun
#' @method box_cox_fun numeric
#' @export
#'
box_cox_fun.numeric <- function(y, simplify = TRUE, ...) {
  simplify_conditionally(lapply(y, FUN = function(yy) {
    if (isTRUE(all.equal(yy, 0L)))
      base::log
    else
      function(x) x ^ yy / yy
  }), simplify = simplify)
}


################################################################################


#' Box-Cox transformation
#'
#' Box-Cox transformation(s) for one to several values of \sQuote{y}.
#'
#' @param x Object convertible to numeric vector. Data to be transformed.
#'   Matrix dimensions, if any, are respected.
#' @param y Object convertible to a numeric vector. Box-Cox parameter(s).
#' @param simplify Logical scalar. Return vector or matrix if resulting list
#'   contained only a single such value?
#' @param ... Optional arguments passed to and from other methods.
#' @family transformation-functions
#' @keywords manip
#' @export
#' @return Transformed values or list of such.
#' @references \url{http://en.wikipedia.org/wiki/Box-Cox_transformation}
#' @references Legendre, P., Legendre, L. 1998 \emph{Numerical Ecology.}
#'   Amsterdam: Elsevier, 853 p.
#' @examples
#' x <- c(0.25, 0.5, 0.75, 1.0)
#' (y <- box_cox(x, 1))
#' stopifnot(y == x)
#' (y <- box_cox(x, 1, simplify = FALSE))
#' stopifnot(is.list(y), length(y) == 1, y[[1]] == x)
#' x <- matrix(x)
#' (y <- box_cox(x, 1:2))
#' stopifnot(is.list(y), length(y) == 2, sapply(y, class) == "matrix")
#' stopifnot(y[[1]] == x, y[[2]] != x)
#'
box_cox <- function(x, y, ...) UseMethod("box_cox")

#' @rdname box_cox
#' @method box_cox numeric
#' @export
#'
box_cox.numeric <- function(x, y, simplify = TRUE, ...) {
  simplify_conditionally(lapply(box_cox_fun(y, simplify = FALSE),
    FUN = function(fun) fun(x)), simplify = simplify)
}

#' @rdname box_cox
#' @method box_cox default
#' @export
#'
box_cox.default <- function(x, y, simplify = TRUE, ...) {
  box_cox(as.numeric(x), y = y, simplify = simplify, ...)
}

#' @rdname box_cox
#' @method box_cox matrix
#' @export
#'
box_cox.matrix <- function(x, y, simplify = TRUE, ...) {
  transform <- function(fun) {
    x[] <- fun(as.vector(x))
    x
  }
  simplify_conditionally(lapply(box_cox_fun(y, simplify = FALSE),
    FUN = transform), simplify = simplify)
}


################################################################################




