
################################################################################


#' Arcsine transformation
#'
#' Conduct arcsine square-root transformations.
#'
#' @param x Numeric vector or convertible to such, or matrix. Data to be
#'   transformed.
#' @param percent Logical scalar. Assume percent values?
#' @inheritParams box_cox
#' @return Numeric vector or matrix of transformed values.
#' @family transformation-functions
#' @keywords manip
#' @export
#' @references Legendre, P., Legendre, L. 1998 \emph{Numerical Ecology.}
#'   Amsterdam: Elsevier, 853 p.
#' @examples
#' x <- c(0.0, 0.25, 0.5, 0.75, 1.0)
#' (y <- asqr(x, percent = FALSE))
#' stopifnot(is.numeric(y), length(y) == length(x), y >= x)
#' x <- matrix(x)
#' (y <- asqr(x, percent = FALSE))
#' stopifnot(is.matrix(y), dim(y) == dim(x), y >= x)
#'
asqr <- function(x, ...) UseMethod("asqr")

#' @rdname asqr
#' @method asqr numeric
#' @export
#'
asqr.numeric <- function(x, percent = TRUE, ...) {
  if (TF(percent)) {
    if (any(x < 0, na.rm = TRUE) || any(x > 100, na.rm = TRUE))
      warning("in 'percent' mode, 'x' should be between 0 and 100")
    100 * asin(sqrt(x / 100))
  } else
    asin(sqrt(x))
}

#' @rdname asqr
#' @method asqr default
#' @export
#'
asqr.default <- function(x, ...) {
  asqr(as.numeric(x), ...)
}

#' @rdname asqr
#' @method asqr matrix
#' @export
#'
asqr.matrix <- function(x, ...) {
  x[] <- asqr(as.vector(x), ...)
  x
}


################################################################################


